<?php
namespace MyApp;

class Subscription
{
    private $db;

    public function __construct($db)
    {
        $this->db = $db;
    }

    /**
     * Handle charging based on user's subscription if available
     *
     * @param int $user_id User ID
     * @param string $idTag Charger ID tag
     * @param int $transactionId Transaction ID
     * @param float $energy Energy used in this increment (in kWh or Wh based on unit)
     * @param string $unit Energy unit (Wh or kWh)
     * @param float $price_per_unit Price per unit of energy
     * @return array Result with pricing details and subscription usage
     */
    public function handleChargingWithSubscription($user_id, $idTag, $transactionId, $energy, $unit, $price_per_unit)
    {
        $result = [
            'subscription_used' => false,
            'subscription_id' => null,
            'subscription_kwh_used' => 0,
            'balance_charge' => 0,
            'energy_unit' => $unit,
            'payment_type' => 'wallet' // Default to wallet payment
        ];

        // Skip if no energy used
        if ($energy <= 0) {
            return $result;
        }

        // Get charger company ID for subscription matching
        $charger = $this->db->row("SELECT company_id FROM `my_charger` WHERE idTag = '{$idTag}'");
        $charger_company_id = $charger['company_id'] ?? 0;

        if ($charger_company_id <= 0) {
            // No company ID, charge directly from wallet
            $result['balance_charge'] = $energy * $price_per_unit;
            return $result;
        }

        // Convert energy to kWh for subscription calculation if needed
        $energy_kwh = $energy;
        if ($unit === 'Wh') {
            $energy_kwh = $energy / 1000;
        }

        // Find active subscription for this user with the same company as the charger
        $subscription = $this->db->row("
            SELECT su.*, s.company_id, s.kwh_allowance, s.name
            FROM subscriptions_users su
            JOIN subscriptions s ON su.subscription_id = s.id
            WHERE su.user_id = '{$user_id}'
            AND su.is_active = 1
            AND su.remaining_kwh > 0
            AND s.company_id = '{$charger_company_id}'
            AND su.expiration_date >= NOW()
            ORDER BY su.id DESC
            LIMIT 1
        ");

        // If no direct subscription, check for fleet-based subscription (automatic inheritance)
        if (!$subscription) {
            $subscription = $this->getFleetSubscription($user_id, $charger_company_id);
        }

        if (!$subscription) {
            // No matching subscription, charge directly from wallet
            $result['balance_charge'] = $energy * $price_per_unit;
            return $result;
        }

        // We found a subscription
        $subscription_id = $subscription['subscription_id'];
        $user_subscription_id = $subscription['id'];
        $available_kwh = $subscription['remaining_kwh'];

        if ($available_kwh >= $energy_kwh) {
            // Full amount can be covered by subscription
            $new_remaining = $available_kwh - $energy_kwh;

            // Update subscription remaining kWh
            $this->db->query("
                UPDATE subscriptions_users
                SET remaining_kwh = '{$new_remaining}',
                    updated_at = NOW()
                WHERE id = '{$user_subscription_id}'
            ");

            $result['subscription_used'] = true;
            $result['subscription_id'] = $subscription_id;
            $result['subscription_kwh_used'] = $energy_kwh;
            $result['payment_type'] = 'subscription';
        } else if ($available_kwh > 0) {
            // Partial amount can be covered by subscription
            $subscription_kwh = $available_kwh;

            // Calculate remaining amount to charge from balance
            $remaining_energy = $energy_kwh - $available_kwh;
            $remaining_in_original_unit = $remaining_energy;
            if ($unit === 'Wh') {
                $remaining_in_original_unit = $remaining_energy * 1000;
            }

            // Update price to charge from balance
            $balance_charge = $remaining_in_original_unit * $price_per_unit;

            // Update subscription - set to 0 as we've used all available kWh
            $this->db->query("
                UPDATE subscriptions_users
                SET remaining_kwh = 0,
                    updated_at = NOW()
                WHERE id = '{$user_subscription_id}'
            ");

            $result['subscription_used'] = true;
            $result['subscription_id'] = $subscription_id;
            $result['subscription_kwh_used'] = $subscription_kwh;
            $result['balance_charge'] = $balance_charge;
            $result['payment_type'] = 'mixed';
        }

        return $result;
    }

    /**
     * Get fleet-based subscription for a user (automatic inheritance)
     * Fleet drivers automatically get access to their fleet's private subscription
     *
     * @param int $user_id User ID
     * @param int $charger_company_id Company ID of the charger
     * @return array|null Subscription data or null if not found
     */
    private function getFleetSubscription($user_id, $charger_company_id)
    {
        // Check if user is part of a fleet that has a private subscription with this company
        $fleetSubscription = $this->db->row("
            SELECT
                s.id as subscription_id,
                s.company_id,
                s.kwh_allowance,
                s.name,
                fu.fleet_id,
                f.name as fleet_name
            FROM fleet_users fu
            JOIN fleets f ON fu.fleet_id = f.id
            JOIN subscriptions s ON s.target_fleet_id = f.id
            WHERE fu.user_id = '{$user_id}'
            AND s.company_id = '{$charger_company_id}'
            AND s.status = 'active'
            AND s.visibility = 'private'
            AND f.archived_at IS NULL
            ORDER BY s.id DESC
            LIMIT 1
        ");

        if (!$fleetSubscription) {
            return null;
        }

        // Check if user already has this subscription assigned
        $existingUserSub = $this->db->row("
            SELECT id, remaining_kwh, expiration_date
            FROM subscriptions_users
            WHERE user_id = '{$user_id}'
            AND subscription_id = '{$fleetSubscription['subscription_id']}'
            AND is_active = 1
            LIMIT 1
        ");

        if ($existingUserSub) {
            // User already has this subscription, return it with remaining_kwh
            return array_merge($fleetSubscription, [
                'id' => $existingUserSub['id'],
                'remaining_kwh' => $existingUserSub['remaining_kwh'],
                'expiration_date' => $existingUserSub['expiration_date']
            ]);
        }

        // Auto-assign the fleet subscription to the user
        $expirationDate = date('Y-m-d H:i:s', strtotime('+1 month'));
        $this->db->query("
            INSERT INTO subscriptions_users (
                user_id, subscription_id, subscription_date, expiration_date,
                is_active, payment_status, remaining_kwh, fleet_sponsored, fleet_id,
                created_at, updated_at
            ) VALUES (
                '{$user_id}',
                '{$fleetSubscription['subscription_id']}',
                NOW(),
                '{$expirationDate}',
                1,
                'paid',
                '{$fleetSubscription['kwh_allowance']}',
                1,
                '{$fleetSubscription['fleet_id']}',
                NOW(),
                NOW()
            )
        ");

        $newSubId = $this->db->lastInsertId();

        return array_merge($fleetSubscription, [
            'id' => $newSubId,
            'remaining_kwh' => $fleetSubscription['kwh_allowance'],
            'expiration_date' => $expirationDate
        ]);
    }
}
