<?php
namespace MyApp;

class Session
{
    private $db;

    public function __construct($db)
    {
        $this->db = $db;
    }

    /**
     * Update session status by transactionId
     */
    public function updateSessionStatus($data = null, $transactionId = null, $status = null)
    {
        if (is_numeric($data) && $transactionId === null) {
            $transactionId = $data;
        }

        if (!$transactionId || !$status) {
            return;
        }

        // When finishing a session, capture the user's current balance
        $currentBalanceClause = "";
        if ($status === 'finished') {
            $session = $this->db->row("SELECT user_id FROM `my_sessions` WHERE transactionId = '{$transactionId}' LIMIT 1");
            if ($session && !empty($session['user_id'])) {
                $user = $this->db->row("SELECT balanse FROM `my_user` WHERE id = '{$session['user_id']}' LIMIT 1");
                if ($user) {
                    $currentBalanceClause = ", current_balance = '{$user['balanse']}'";
                }
            }
        }

        $this->db->query("
            UPDATE `my_sessions`
            SET status = '{$status}',
                updated_at = NOW()
                {$currentBalanceClause}
            WHERE transactionId = '{$transactionId}'
            LIMIT 1
        ");
    }

    public function endSessionIfExists($transactionId, $idTag = null, $connectorId = null)
    {
        $session = $this->db->row("
            SELECT s.*, fu.charging_limit, fu.billing_type
            FROM `my_sessions` s
            LEFT JOIN fleet_users fu ON s.fleet_id = fu.fleet_id AND s.user_id = fu.user_id
            WHERE s.transactionId = '{$transactionId}'
            LIMIT 1
        ");

        if ($session) {
            // Prevent duplicate processing - only process if not already finished
            if ($session['status'] === 'finished') {
                error_log("endSessionIfExists: Session already finished for transactionId={$transactionId}, skipping");
                return;
            }

            // Attempt session fee deduction before finishing — catches cases where
            // StopTransaction arrived before any MeterValues triggered deductSessionFee()
            if (!$session['session_fee_paid'] && (float)($session['session_fee'] ?? 0) > 0) {
                $sessionIdTag = $session['idTag'] ?? $idTag;
                $sessionConnectorId = $session['connectorId'] ?? $connectorId;
                if ($sessionIdTag && $sessionConnectorId) {
                    error_log("endSessionIfExists: Session fee unpaid for transactionId={$transactionId}, attempting deduction before finishing");
                    $this->deductSessionFee($transactionId, $sessionIdTag, $sessionConnectorId);
                }
            }

            $this->updateSessionStatus($transactionId, null, 'finished');

            // Record charging usage for fleet drivers (both with and without limits)
            // This tracks usage for all fleet sessions, not just those with limits
            if (!empty($session['fleet_id']) && !empty($session['user_id'])) {
                // Calculate total from operations table to avoid race condition
                // The session's total_cost might not be updated yet when StopTransaction is received
                $operationsTotal = $this->db->row("
                    SELECT COALESCE(SUM(money), 0) as total_cost
                    FROM my_user_operations
                    WHERE transactionId = '{$transactionId}'
                    AND user_id = '{$session['user_id']}'
                    AND type = 'charger'
                ");

                $energyCost = (float) ($operationsTotal['total_cost'] ?? 0);
                $sessionFee = (float) ($session['session_fee'] ?? 0);
                $sessionFeePaid = (int) ($session['session_fee_paid'] ?? 0);

                // Only include session fee if it was actually paid
                $totalAmount = $energyCost + ($sessionFeePaid ? $sessionFee : 0);

                error_log("endSessionIfExists: transactionId={$transactionId}, user={$session['user_id']}, fleet={$session['fleet_id']}, energyCost={$energyCost}, sessionFee={$sessionFee}, sessionFeePaid={$sessionFeePaid}, totalAmount={$totalAmount}");

                if ($totalAmount > 0) {
                    $this->recordDriverChargingUsage($session['user_id'], $session['fleet_id'], $totalAmount);
                } else {
                    error_log("endSessionIfExists: Total amount is 0 or negative, not recording usage");
                }
            } else {
                error_log("endSessionIfExists: Not recording usage - fleet_id: " . ($session['fleet_id'] ?? 'null') . ", user_id: " . ($session['user_id'] ?? 'null'));
            }

            // Queue notification for Laravel to process
            $this->queueSessionCompletedNotification($session);
        } else {
            error_log("endSessionIfExists: No session found for transactionId={$transactionId}");
        }
    }

    /**
     * Queue session completed notification for Laravel to process
     */
    private function queueSessionCompletedNotification($session)
    {
        try {
            $now = date('Y-m-d H:i:s');
            $payload = json_encode([
                'session_id' => $session['id'],
                'energy_kwh' => $session['total_energy'] ?? null,
                'amount' => $session['total_cost'] ?? null,
                'receipt_number' => $session['transactionId'] ?? null,
            ]);

            $this->db->query("
                INSERT INTO `notification_queue` (`type`, `payload`, `status`, `created_at`, `updated_at`)
                VALUES ('session_completed', '{$payload}', 'pending', '{$now}', '{$now}')
            ");

            error_log("Session completed notification queued for session {$session['id']}");
        } catch (\Exception $e) {
            error_log("Error queueing session completed notification: " . $e->getMessage());
        }
    }

    /**
     * Record charging usage for a fleet driver
     */
    private function recordDriverChargingUsage($userId, $fleetId, $amount)
    {
        if (!$userId || !$fleetId || $amount <= 0) {
            error_log("recordDriverChargingUsage: Invalid params - userId: {$userId}, fleetId: {$fleetId}, amount: {$amount}");
            return false;
        }

        error_log("recordDriverChargingUsage: Updating usage - userId: {$userId}, fleetId: {$fleetId}, amount: {$amount}");

        $this->db->query("
            UPDATE fleet_users
            SET current_period_usage = IFNULL(current_period_usage, 0) + {$amount},
                updated_at = NOW()
            WHERE fleet_id = '{$fleetId}'
            AND user_id = '{$userId}'
        ");

        error_log("recordDriverChargingUsage: Update completed");

        return true;
    }

    public function setTransactionIdForSession($transactionId, $idTag, $connectorId) {
        if (!isset($transactionId)) {
            return;
        }

        // First try exact match (idTag + connectorId)
        $affectedRows = $this->db->query("
            UPDATE `my_sessions`
            SET transactionId = '{$transactionId}',
                connectorId = '{$connectorId}'
            WHERE idTag = '{$idTag}'
            AND connectorId = '{$connectorId}'
            AND transactionId IS NULL
            AND status = 'pending'
            ORDER BY id DESC
            LIMIT 1
        ");

        if ($affectedRows == 0) {
            // No exact match found - try matching by user_id + idTag
            // Get user_id from command buffer to ensure we match the correct user's session
            $buffer = $this->db->row("SELECT user_id FROM my_set_command_buffer WHERE idTag = '{$idTag}' LIMIT 1");

            if ($buffer && !empty($buffer['user_id'])) {
                $userId = $buffer['user_id'];
                error_log("setTransactionIdForSession: No exact match for idTag={$idTag}, connectorId={$connectorId}. Trying user_id={$userId} + idTag match.");

                $affectedRows = $this->db->query("
                    UPDATE `my_sessions`
                    SET transactionId = '{$transactionId}',
                        connectorId = '{$connectorId}'
                    WHERE idTag = '{$idTag}'
                    AND user_id = '{$userId}'
                    AND transactionId IS NULL
                    AND status = 'pending'
                    ORDER BY id DESC
                    LIMIT 1
                ");

                if ($affectedRows > 0) {
                    error_log("setTransactionIdForSession: Updated session using user_id + idTag match. transactionId={$transactionId}, user_id={$userId}, connectorId updated to {$connectorId}");
                } else {
                    error_log("setTransactionIdForSession: No pending session found for idTag={$idTag}, user_id={$userId}");
                }
            } else {
                // Fallback to idTag-only if no user in buffer (shouldn't happen normally)
                error_log("setTransactionIdForSession: No user_id in buffer for idTag={$idTag}. Falling back to idTag-only match.");

                $affectedRows = $this->db->query("
                    UPDATE `my_sessions`
                    SET transactionId = '{$transactionId}',
                        connectorId = '{$connectorId}'
                    WHERE idTag = '{$idTag}'
                    AND transactionId IS NULL
                    AND status = 'pending'
                    ORDER BY id DESC
                    LIMIT 1
                ");

                if ($affectedRows > 0) {
                    error_log("setTransactionIdForSession: Updated session using idTag-only match. transactionId={$transactionId}, connectorId updated to {$connectorId}");
                } else {
                    error_log("setTransactionIdForSession: No pending session found for idTag={$idTag}");
                }
            }
        } else {
            error_log("setTransactionIdForSession: Updated session with exact match. transactionId={$transactionId}, idTag={$idTag}, connectorId={$connectorId}");
        }
    }

    public function setSessionStarting($transactionId, $idTag, $connectorId)
    {
        $this->db->query("
        UPDATE `my_sessions`
        SET status = 'starting', updated_at = NOW()
        WHERE transactionId = '{$transactionId}'
          AND idTag = '{$idTag}'
          AND connectorId = '{$connectorId}'
        ORDER BY id DESC
        LIMIT 1
    ");
    }

    public function setSessionCharging($transactionId, $idTag, $connectorId)
    {
        $this->db->query("
        UPDATE `my_sessions`
        SET status = 'charging', updated_at = NOW()
            WHERE transactionId = '{$transactionId}'
              AND idTag = '{$idTag}'
              AND connectorId = '{$connectorId}'
            ORDER BY id DESC
            LIMIT 1
        ");

        $this->deductSessionFee($transactionId, $idTag, $connectorId);
    }

    public function updateChargingValues($transactionId, $idTag, $connectorId, $thisH, $price_m) {
        $this->db->query("
        UPDATE `my_sessions`
        SET total_energy = IFNULL(total_energy, 0) + '{$thisH}',
            total_cost = IFNULL(total_cost, 0) + '{$price_m}',
            updated_at = NOW()
        WHERE transactionId = '{$transactionId}'
        AND idTag = '{$idTag}'
        AND connectorId = '{$connectorId}'
        ORDER BY id DESC
        LIMIT 1
    ");
    }

    private function deductSessionFee($transactionId, $idTag, $connectorId)
    {
        $session = $this->db->row("
            SELECT s.*, fu.billing_type, f.balance as fleet_balance
            FROM my_sessions s
            LEFT JOIN fleet_users fu ON s.fleet_id = fu.fleet_id AND s.user_id = fu.user_id
            LEFT JOIN fleets f ON s.fleet_id = f.id
            WHERE s.transactionId = '{$transactionId}'
              AND s.idTag = '{$idTag}'
              AND s.connectorId = '{$connectorId}'
            ORDER BY s.id DESC
            LIMIT 1
        ");

        if (!$session || $session['session_fee_paid']) {return;} // EARLY RETURN

        $userId = $session['user_id'] ?? null;
        $sessionFee = (float) ($session['session_fee'] ?? 0);
        $fleetId = $session['fleet_id'] ?? null;
        $billingType = $session['billing_type'] ?? 'individual';

        if (!$userId) {return;}

        // Case 1: Free session, always mark as paid
        if ($sessionFee == 0) {
            $this->db->query("
            UPDATE my_sessions
            SET session_fee_paid = 1, updated_at = NOW()
            WHERE id = '{$session['id']}'
            LIMIT 1
        ");
            return;
        }

        $now = time();

        // Case 2: Fleet corporate billing - deduct from fleet balance
        if ($fleetId && $billingType === 'corporate') {
            $fleetBalance = (float) ($session['fleet_balance'] ?? 0);
            
            if ($fleetBalance >= $sessionFee) {
                $newFleetBalance = $fleetBalance - $sessionFee;
                
                // Update fleet balance
                $this->db->query("UPDATE fleets SET balance = '{$newFleetBalance}' WHERE id = '{$fleetId}'");
                
                // Insert session_fee operation with fleet info
                $this->db->query("
                    INSERT INTO my_user_operations
                    (user_id, transactionId, operationTime, MeterValues, money, type, payment_type, energy_unit, user_balanse, fleet_id, billing_type)
                    VALUES
                    ('{$userId}', '{$transactionId}', '{$now}', 0, '{$sessionFee}', 'session_fee', 'fleet_corporate', NULL, '{$newFleetBalance}', '{$fleetId}', 'fleet_corporate')
                ");
                

                // Mark session as paid
                $this->db->query("
                UPDATE my_sessions
                SET session_fee_paid = 1, updated_at = NOW()
                WHERE id = '{$session['id']}'
                LIMIT 1
            ");
            }
        } else {
            // Case 3: Individual billing (regular or fleet individual)
            $user = $this->db->row("
                SELECT balanse FROM my_user
                WHERE id = '{$userId}' AND visible = 1
                LIMIT 1
            ");

            if (!$user) { return; } // EARLY RETURN

            $userBalance = (float) $user['balanse'];
            $newBalance = $userBalance - $sessionFee;

            // Deduct fee as long as the user can cover it (>= 0 after deduction).
            // The GHS 5 minimum balance was already enforced when the session was started.
            if ($newBalance >= 0) {
                $paymentType = $fleetId ? 'fleet_individual' : 'wallet';
                $billingTypeDb = $fleetId ? 'fleet_individual' : 'wallet';

                // Insert session_fee operation
                $this->db->query("
                    INSERT INTO my_user_operations
                    (user_id, transactionId, operationTime, MeterValues, money, type, payment_type, energy_unit, user_balanse, fleet_id, billing_type)
                    VALUES
                    ('{$userId}', '{$transactionId}', '{$now}', 0, '{$sessionFee}', 'session_fee', '{$paymentType}', NULL, '{$newBalance}', " . ($fleetId ? "'{$fleetId}'" : 'NULL') . ", '{$billingTypeDb}')
                ");

                // Deduct from balance
                $this->db->query("
                    UPDATE my_user
                    SET balanse = '{$newBalance}'
                    WHERE id = '{$userId}'
                    LIMIT 1
                ");

                // Mark session as paid
                $this->db->query("
                UPDATE my_sessions
                SET session_fee_paid = 1, updated_at = NOW()
                WHERE id = '{$session['id']}'
                LIMIT 1
            ");
            }
        }
    }

    public function linkUserToCompany($idTag, $userId)
    {
        $charger = $this->db->row("SELECT company_id FROM my_charger WHERE idTag = '{$idTag}' LIMIT 1");

        if ($charger && $charger['company_id']) {
            $exists = $this->db->row("
            SELECT id FROM my_company_customers
            WHERE company_id = '{$charger['company_id']}'
              AND user_id = '{$userId}'
            LIMIT 1
        ");

            if (!$exists) {
                $now = time();
                $this->db->query("
                INSERT INTO my_company_customers (company_id, user_id, created_at, updated_at)
                VALUES ('{$charger['company_id']}', '{$userId}', FROM_UNIXTIME({$now}), FROM_UNIXTIME({$now}))
            ");
            }
        }
    }

}
